namespace Microsoft.Samples.PlanMyNight.Web.Controllers.Tests
{
    using System;
    using System.Collections.Specialized;
    using System.ComponentModel.Composition;
    using System.ComponentModel.Composition.Hosting;
    using System.Linq;
    using System.Security.Principal;
    using System.Web.Mvc;
    using System.Web.Routing;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Infrastructure;
    using Microsoft.Samples.PlanMyNight.Web.Controllers;
    using Microsoft.Samples.PlanMyNight.Web.ViewModels;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;
    using MvcContrib.TestHelper;
    using Rhino.Mocks;

    [TestClass]
    public class ItinerariesControllerFixture
    {
        [TestInitialize]
        public void InitializeServiceLocator()
        {
            ServiceLocator.SetLocatorProvider(() => new CompositionContainer(new TypeCatalog(typeof(MockActionExport))));
        }

        [TestMethod]
        public void CurrentWithEmptyContainerReturnsEmpty()
        {
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(default(Itinerary));

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object, container.Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var result = controller.Current();
            var viewName = result.AssertResultIs<PartialViewResult>().ViewName;
            var model = result.AssertResultIs<PartialViewResult>().ViewData.Model as ItineraryDetailsViewModel;

            Assert.AreEqual("Itinerary", viewName);
            Assert.IsNotNull(model);
            Assert.AreEqual(0, model.ActivitiesCount);
            container.VerifyAll();
        }

        [TestMethod]
        public void Index()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary1 = new Itinerary() { Name = "My Night", Id = 11, Created = DateTime.Now.AddHours(-1) };
            itinerary1.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary1.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary1.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });
            var itinerary2 = new Itinerary() { Name = "My Night (copy)", Id = 12, Created = DateTime.Now };
            itinerary2.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary2.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary2.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });

            var container = new Mock<IItineraryContainer>();

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RetrieveByUser(new Guid(UserId)))
                .Returns(new[] { itinerary1, itinerary2 });

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, container.Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var viewData = controller.Index().AssertViewRendered().ViewData;
            var model = viewData.Model as ItinerariesViewModel;

            Assert.IsNotNull(viewData);
            Assert.IsNotNull(model);
            Assert.AreEqual(2, model.Itineraries.Count());
            repo.VerifyAll();
        }

        [TestMethod]
        public void SaveCurrentItinerary()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { UserId = new Guid(UserId) };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "10", Order = 2 });
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            container.Setup(o => o.SetCurrent(itinerary));

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Add(It.IsAny<Itinerary>()))
                .Callback((Itinerary i) => i.Id = 11);
            repo.Setup(o => o.Retrieve(11L))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, container.Object);
            builder.InitializeController(controller);

            controller.Save("Night Out", "Description", true).AssertActionRedirect()
                                        .ToAction("Details")
                                        .WithParameter("id", 11L);

            container.VerifyAll();
            repo.Verify(o => o.Add(It.Is<Itinerary>(i => i.Name == "Night Out")));
            repo.Verify(o => o.Add(It.Is<Itinerary>(i => i.Description == "Description")));
            repo.Verify(o => o.Add(It.Is<Itinerary>(i => i.IsPublic)));
            repo.Verify(o => o.Retrieve(11L));
        }

        [TestMethod]
        public void CurrentReturnsItineraryInModel()
        {
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(new Itinerary());

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object, container.Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var result = controller.Current();
            var viewName = result.AssertResultIs<PartialViewResult>().ViewName;
            var model = result.AssertResultIs<PartialViewResult>().ViewData.Model as ItineraryDetailsViewModel;

            Assert.AreEqual("Itinerary", viewName);
            Assert.IsNotNull(model);
            container.VerifyAll();
        }

        [TestMethod]
        public void AddActivityToEmptyContainerCreatesItinerary()
        {
            Itinerary itinerary = null;
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(default(Itinerary));
            container.Setup(o => o.SetCurrent(It.IsAny<Itinerary>())).Callback((Itinerary i) => itinerary = i);

            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivity("6")).Returns(new Activity
            {
                Id = "6",
                Name = "Test Activity",
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Location = new[] { 48.545691, -117.891798 },
                Zip = "99114"
            });

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.AddActivity("6");

            result.AssertHttpRedirect();
            container.VerifyAll();
            repo.VerifyAll();
            Assert.IsNotNull(itinerary);
            Assert.AreEqual(1, itinerary.Activities.Count);
        }

        [TestMethod]
        public void AddActivityToItineraryContainer()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            container.Setup(o => o.SetCurrent(itinerary));

            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivity("6")).Returns(new Activity
            {
                Id = "6",
                Name = "Test Activity",
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Location = new[] { 48.545691, -117.891798 },
                Zip = "99114"
            });

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.AddActivity("6");

            result.AssertHttpRedirect();
            container.VerifyAll();
            repo.VerifyAll();
            Assert.AreEqual(2, itinerary.Activities.Count);
            Assert.AreEqual(1, itinerary.Activities.Where(a => a.ActivityId == "6").SingleOrDefault().Order);
        }

        [TestMethod]
        public void AddActivityWrongIdRedirectsBack()
        {
            var itinerary = new Itinerary();
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivity("6")).Returns(default(Activity));

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.AddActivity("6");

            result.AssertHttpRedirect();
            container.VerifyAll();
            repo.VerifyAll();
            Assert.AreEqual(0, itinerary.Activities.Count);
        }

        [TestMethod]
        public void AddSameActivityTwiceDoesNothing()
        {
            var selectedActivity = new Activity { Id = "6", Name = "Test Activity" };
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = selectedActivity.Id, Activity = selectedActivity });
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.AddActivity("6");

            result.AssertHttpRedirect();
            container.VerifyAll();
            Assert.AreEqual(1, itinerary.Activities.Count);
        }

        [TestMethod]
        public void AddActivitiesAppendsToCurrentItinerary()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "2", Order = 2 });
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            container.Setup(o => o.SetCurrent(itinerary));

            var repo = new Mock<IActivitiesRepository>();
            repo.Setup(o => o.RetrieveActivity("3")).Returns(new Activity
            {
                Id = "3",
                Name = "Test Activity #3",
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Location = new[] { 48.545691, -117.891798 },
                Zip = "99114"
            });
            repo.Setup(o => o.RetrieveActivity("4")).Returns(new Activity
            {
                Id = "4",
                Name = "Test Activity #4",
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Location = new[] { 48.545691, -117.891798 },
                Zip = "99114"
            });
            repo.Setup(o => o.RetrieveActivity("5")).Returns(new Activity
            {
                Id = "5",
                Name = "Test Activity #5",
                TypeId = 1,
                State = "WA",
                City = "Colville",
                Location = new[] { 48.545691, -117.891798 },
                Zip = "99114"
            });

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.AddActivities("3|4|5");

            result.AssertHttpRedirect();
            container.VerifyAll();
            repo.VerifyAll();
            Assert.AreEqual(5, itinerary.Activities.Count);
            Assert.AreEqual(2, itinerary.Activities.Where(a => a.ActivityId == "3").SingleOrDefault().Order);
            Assert.AreEqual(3, itinerary.Activities.Where(a => a.ActivityId == "4").SingleOrDefault().Order);
            Assert.AreEqual(4, itinerary.Activities.Where(a => a.ActivityId == "5").SingleOrDefault().Order);
        }

        [TestMethod]
        public void DeleteActivityEmptyContainerRedirectsBack()
        {
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(default(Itinerary));

            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.DeleteActivity("6");

            result.AssertHttpRedirect();
            container.VerifyAll();
        }

        [TestMethod]
        public void DeleteActivityWrongIdRedirectsBack()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1" });
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.DeleteActivity("6");

            result.AssertHttpRedirect();
            container.VerifyAll();
            Assert.AreEqual(1, itinerary.Activities.Count);
        }

        [TestMethod]
        public void DeleteActivityUpdatesItinerary()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1" });
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.DeleteActivity("1");

            result.AssertHttpRedirect();
            container.VerifyAll();
            Assert.AreEqual(0, itinerary.Activities.Count);
        }

        [TestMethod]
        public void MoveActivityNullItineraryDoesNothing()
        {
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(default(Itinerary));
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MoveActivity("1", 1);

            result.AssertHttpRedirect();
            container.VerifyAll();
        }

        [TestMethod]
        public void MoveUnexistingActivityDoesNothing()
        {
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(new Itinerary());
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MoveActivity("1", 1);

            result.AssertHttpRedirect();
            container.VerifyAll();
        }

        [TestMethod]
        public void MoveActivityDown()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 0 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "2", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "3", Order = 2 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "4", Order = 3 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MoveActivity("3", 3);

            result.AssertHttpRedirect();
            container.VerifyAll();
            var activities = itinerary.Activities.OrderBy(a => a.Order).ToArray();
            Assert.AreEqual("1", activities[0].ActivityId);
            Assert.AreEqual("2", activities[1].ActivityId);
            Assert.AreEqual("4", activities[2].ActivityId);
            Assert.AreEqual("3", activities[3].ActivityId);
        }

        [TestMethod]
        public void MoveActivityUp()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 0 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "2", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "3", Order = 2 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "4", Order = 3 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MoveActivity("3", 1);

            result.AssertHttpRedirect();
            container.VerifyAll();
            var activities = itinerary.Activities.OrderBy(a => a.Order).ToArray();
            Assert.AreEqual("1", activities[0].ActivityId);
            Assert.AreEqual("3", activities[1].ActivityId);
            Assert.AreEqual("2", activities[2].ActivityId);
            Assert.AreEqual("4", activities[3].ActivityId);
        }

        [TestMethod]
        public void MoveTopActivityUpDoesNothing()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 0 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "2", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "3", Order = 2 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "4", Order = 3 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MoveActivity("1", 0);

            result.AssertHttpRedirect();
            container.VerifyAll();
            var activities = itinerary.Activities.OrderBy(a => a.Order).ToArray();
            Assert.AreEqual("1", activities[0].ActivityId);
            Assert.AreEqual("2", activities[1].ActivityId);
            Assert.AreEqual("3", activities[2].ActivityId);
            Assert.AreEqual("4", activities[3].ActivityId);
        }

        [TestMethod]
        public void MoveBottomActivityDownDoesNothing()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 0 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "2", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "3", Order = 2 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "4", Order = 3 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MoveActivity("4", 3);

            result.AssertHttpRedirect();
            container.VerifyAll();
            var activities = itinerary.Activities.OrderBy(a => a.Order).ToArray();
            Assert.AreEqual("1", activities[0].ActivityId);
            Assert.AreEqual("2", activities[1].ActivityId);
            Assert.AreEqual("3", activities[2].ActivityId);
            Assert.AreEqual("4", activities[3].ActivityId);
        }

        [TestMethod]
        public void NewEmptyItineraryContainer()
        {
            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.SetCurrent(It.Is<Itinerary>(i => i.Activities.Count == 0)));

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var viewResult = controller.New().AssertHttpRedirect();

            container.VerifyAll();
        }

        [TestMethod]
        public void SaveHttpGetReturnsSaveView()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);
            PrepareControllerMock(controller);

            var viewResult = controller.Save().AssertViewRendered();
            Assert.AreEqual("Edit", viewResult.ViewName);
        }

        [TestMethod]
        public void ShouldShowItineraryDetails()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId) };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var result = controller.Details(itinerary.Id).AssertViewRendered();

            var model = result.ViewData.Model as ItineraryDetailsViewModel;
            Assert.IsNotNull(model);
            Assert.IsTrue(model.CanEdit);
            Assert.IsFalse(model.CanRate);
        }

        [TestMethod]
        public void ShouldNotShowItineraryDetailsForDifferentUser()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            controller.Details(itinerary.Id).AssertHttpRedirect();
        }

        [TestMethod]
        public void ShouldAllowEditingItinerary()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId) };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var container = new Mock<IItineraryContainer>();
            Itinerary itineraryAtContainer = null;
            container.Setup(c => c.SetCurrent(It.IsAny<Itinerary>())).Callback<Itinerary>(i => itineraryAtContainer = i);
            container.Setup(c => c.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, container.Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var result = controller.Edit(itinerary.Id).AssertViewRendered();

            Assert.IsNotNull(result.ViewData);
            Assert.IsNotNull(result.ViewData.Model);
            Assert.IsInstanceOfType(result.ViewData.Model, typeof(ItineraryDetailsViewModel));
            Assert.AreEqual(itinerary, itineraryAtContainer);
        }

        [TestMethod]
        public void ShouldNotAllowEditingItineraryForDifferentUser()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            controller.Edit(itinerary.Id).AssertHttpRedirect();
        }

        [TestMethod]
        public void ShouldNotShowItineraryDetailsForGuestUser()
        {
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = null;
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var viewData = controller.Details(itinerary.Id).AssertHttpRedirect();
        }

        [TestMethod]
        public void ShouldShowPublicItineraryDetailsOwnedByDifferentUser()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, IsPublic = true, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);
            repo.Setup(o => o.GetUserDisplayName(It.IsAny<Guid>())).Returns("John Doe");

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Expect(o => o.Url).Return(new Uri("http://planmynight.net/Itineraries/1", UriKind.Absolute));
            var serverVariables = new NameValueCollection();
            serverVariables.Add("HTTP_HOST", "planmynight.net");
            builder.HttpContext.Request.Expect(o => o.ServerVariables).Return(serverVariables);
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var viewData = controller.Details(itinerary.Id).AssertViewRendered().ViewData;

            var model = viewData.Model as ItineraryDetailsViewModel;

            Assert.IsNotNull(model);
            Assert.IsFalse(model.CanEdit);
            Assert.AreEqual("John Doe", model.Owner);
        }

        [TestMethod]
        public void ShouldShowPublicItineraryDetailsForGuestUser()
        {
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, IsPublic = true, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);
            repo.Setup(o => o.GetUserDisplayName(It.IsAny<Guid>())).Returns("John Doe");

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = null;
            builder.HttpContext.Request.Expect(o => o.Url).Return(new Uri("http://planmynight.net/Itineraries/1", UriKind.Absolute));
            var serverVariables = new NameValueCollection();
            serverVariables.Add("HTTP_HOST", "planmynight.net");
            builder.HttpContext.Request.Expect(o => o.ServerVariables).Return(serverVariables);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var viewData = controller.Details(itinerary.Id).AssertViewRendered().ViewData;
            var model = viewData.Model as ItineraryDetailsViewModel;

            Assert.IsNotNull(model);
            Assert.IsFalse(model.CanEdit);
            Assert.IsFalse(model.CanRate);
            Assert.AreEqual("John Doe", model.Owner);
        }

        [TestMethod]
        public void ShouldShowPublicItineraryWithDisplayNameOfOwner()
        {
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, IsPublic = true, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(itinerary.Id)).Returns(itinerary);
            repo.Setup(o => o.GetUserDisplayName(itinerary.UserId)).Returns("Test user name");

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Expect(o => o.Url).Return(new Uri("http://planmynight.net/Itineraries/1", UriKind.Absolute));
            var serverVariables = new NameValueCollection();
            serverVariables.Add("HTTP_HOST", "planmynight.net");
            builder.HttpContext.Request.Expect(o => o.ServerVariables).Return(serverVariables);
            builder.HttpContext.User = null;
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var viewData = controller.Details(itinerary.Id).AssertViewRendered().ViewData;
            var model = viewData.Model as ItineraryDetailsViewModel;

            Assert.IsNotNull(model);
            Assert.AreEqual("Test user name", model.Owner);
        }

        [TestMethod]
        public void ShouldMakeAnItineraryPublic()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId) };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var result = controller.MakePublic(itinerary.Id);

            repo.Verify(r => r.Update(It.Is<Itinerary>(i => i.Id == itinerary.Id && i.IsPublic == true)));
        }

        [TestMethod]
        public void ShouldNotAllowToMakeAnItineraryPublicForDifferentUser()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = Guid.NewGuid() };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            bool itineraryWasUpdated = false;
            repo.Setup(r => r.Update(It.IsAny<Itinerary>())).Callback(() => itineraryWasUpdated = true);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var result = controller.MakePublic(itinerary.Id);

            Assert.IsFalse(itineraryWasUpdated);
        }

        [TestMethod]
        public void ShouldRedirectToItineraryAfterMakingItPublic()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId) };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var result = controller.MakePublic(itinerary.Id);

            result.AssertActionRedirect().ToAction("Details").WithParameter("id", itinerary.Id);
        }

        [TestMethod]
        public void ShouldMakeAnItineraryPrivate()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId), IsPublic = true };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var result = controller.MakePrivate(itinerary.Id);

            repo.Verify(r => r.Update(It.Is<Itinerary>(i => i.Id == itinerary.Id && i.IsPublic == false)));
        }

        [TestMethod]
        public void ShouldMakeAnItineraryPrivateAndUpdateCurrentIfSame()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId), IsPublic = true };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent())
                     .Returns(new Itinerary { Id = 11, IsPublic = true });
            container.Setup(o => o.SetCurrent(It.Is<Itinerary>(i => i.IsPublic == false)));

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.MakePrivate(itinerary.Id);

            repo.Verify(r => r.Update(It.Is<Itinerary>(i => i.Id == itinerary.Id && i.IsPublic == false)));
            container.VerifyAll();
        }

        [TestMethod]
        public void ShouldNotAllowToMakeAnItineraryPrivateForDifferentUser()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = Guid.NewGuid(), IsPublic = true };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            bool itineraryWasUpdated = false;
            repo.Setup(r => r.Update(It.IsAny<Itinerary>())).Callback(() => itineraryWasUpdated = true);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var result = controller.MakePrivate(itinerary.Id);

            Assert.IsFalse(itineraryWasUpdated);
        }

        [TestMethod]
        public void ShouldRedirectToItineraryAfterMakingItPrivate()
        {
            const string UserId = "0CA4C8ED-86D5-402e-9BE0-80391CC6BC01";
            var itinerary = new Itinerary() { Name = "My Night", Id = 11, UserId = new Guid(UserId), IsPublic = true };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(It.Is<long>(id => id == itinerary.Id)))
                .Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(UserId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var result = controller.MakePrivate(itinerary.Id);

            result.AssertActionRedirect().ToAction("Details").WithParameter("id", itinerary.Id);
        }

        [TestMethod]
        public void SetEstimatedTime()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "123", Order = 0 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            builder.InitializeController(controller);

            var result = controller.SetEstimatedTime("123", 60);

            container.Verify(c => c.SetCurrent(itinerary));
            Assert.AreEqual(60, itinerary.Activities[0].EstimatedMinutes);
        }

        [TestMethod]
        public void SetEstimatedTime_RedirectsBack()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "123", Order = 0 });

            var container = new Mock<IItineraryContainer>();
            container.Setup(o => o.GetCurrent()).Returns(itinerary);
            var repo = new Mock<IActivitiesRepository>();

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(repo.Object, new Mock<IItinerariesRepository>().Object, container.Object);
            var serverVariables = new NameValueCollection();
            serverVariables["HTTP_REFERER"] = "http://returnUrl";
            builder.HttpContext.Request.Stub(r => r.ServerVariables).Return(serverVariables);
            builder.InitializeController(controller);

            var result = controller.SetEstimatedTime("123", 60);

            var redirect = result.AssertHttpRedirect();
            Assert.AreEqual("http://returnUrl", redirect.Url);
        }

        [TestMethod]
        public void CanGetJsonVersionOfItineraryDetails()
        {
            var itinerary = new Itinerary() { Id = 123, IsPublic = true };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "345", Order = 1 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(r => r.Retrieve(123)).Returns(itinerary);

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Expect(o => o.Url).Return(new Uri("http://planmynight.net/Itineraries/1", UriKind.Absolute));
            var serverVariables = new NameValueCollection();
            serverVariables.Add("HTTP_HOST", "planmynight.net");
            builder.HttpContext.Request.Expect(o => o.ServerVariables).Return(serverVariables);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);

            PrepareControllerMock(controller);
            builder.HttpContext.Request.Stub(r => r.ContentType).Return("application/json");
            builder.InitializeController(controller);

            var actual = controller.Details(123);

            var result = actual.AssertResultIs<JsonResult>();
            var jsonModel = result.Data as ItineraryViewModel;
            Assert.IsNotNull(jsonModel);
        }

        [TestMethod]
        public void UserPublicItineraries()
        {
            Guid userId = new Guid("0CA4C8ED-86D5-402e-9BE0-80391CC6BC01");
            var itinerary1 = new Itinerary() { Name = "Private", Id = 11, Created = DateTime.Today, IsPublic = false, UserId = userId };
            itinerary1.Activities.Add(new ItineraryActivity { ActivityId = "1", Order = 1 });
            var itinerary2 = new Itinerary() { Name = "Public", Id = 12, Created = DateTime.Today, IsPublic = true, UserId = userId };
            itinerary2.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RetrieveByUser(userId)).Returns(new[] { itinerary1, itinerary2 });

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Expect(o => o.Url).Return(new Uri("http://planmynight.net/Itineraries/ByUser/1", UriKind.Absolute));
            var serverVariables = new NameValueCollection();
            serverVariables.Add("HTTP_HOST", "planmynight.net");
            builder.HttpContext.Request.Expect(o => o.ServerVariables).Return(serverVariables);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var viewData = controller.ByUser(userId).AssertViewRendered().ViewData;
            var model = viewData.Model as ItinerariesViewModel;

            Assert.IsNotNull(model);
            Assert.AreEqual(1, model.Itineraries.Count());
            Assert.AreEqual(itinerary2.Id, model.Itineraries.ElementAt(0).Id);
        }

        [TestMethod]
        public void UserPublicItinerariesShowsOwnerName()
        {
            Guid userId = new Guid("0CA4C8ED-86D5-402e-9BE0-80391CC6BC01");
            var itinerary = new Itinerary() { Name = "Public", Id = 12, Created = DateTime.Today, IsPublic = true, UserId = userId };
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = "6", Order = 3 });

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RetrieveByUser(userId)).Returns(new[] { itinerary });
            repo.Setup(o => o.GetUserDisplayName(userId)).Returns("Test user name");

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Expect(o => o.Url).Return(new Uri("http://planmynight.net/Itineraries/ByUser/1", UriKind.Absolute));
            var serverVariables = new NameValueCollection();
            serverVariables.Add("HTTP_HOST", "planmynight.net");
            builder.HttpContext.Request.Expect(o => o.ServerVariables).Return(serverVariables);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            PrepareControllerMock(controller);
            builder.InitializeController(controller);

            var viewData = controller.ByUser(userId).AssertViewRendered().ViewData;
            var model = viewData.Model as ItinerariesViewModel;

            Assert.IsNotNull(model);
            Assert.AreEqual("Test user name", model.Owner);
            Assert.IsNotNull(model.Itineraries);
            repo.VerifyAll();
        }

        [TestMethod]
        public void ListCallRepositoryAndReturnsPartialView()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RetrieveComments(It.IsAny<long>()))
                .Returns(new[] { new ItineraryComment(), new ItineraryComment() });

            var builder = new TestControllerBuilder();
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var view = controller.ListComments(10).AssertResultIs<PartialViewResult>();

            Assert.IsNotNull(view.ViewData.Model);
            Assert.IsInstanceOfType(view.ViewData.Model, typeof(ItineraryCommentsViewModel));
            repo.Verify(o => o.RetrieveComments(10));
        }

        [TestMethod]
        public void NewCommentToUnexistingItineraryRedirectsBack()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(0)).Returns(default(Itinerary));

            var builder = new TestControllerBuilder();
            var userId = "BE62FBC7-DED9-4420-913A-F47A5324A1C6";
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(userId), null);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            controller.AddComment(0, "test comment").AssertHttpRedirect();
            repo.VerifyAll();
        }

        [TestMethod]
        public void NewCommentReturnsViewWithActivity()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(3))
                .Returns(new Itinerary() { IsPublic = true });

            var serverVariables = new NameValueCollection();
            serverVariables["REMOTE_ADDR"] = "127.0.0.1";
            var builder = new TestControllerBuilder();
            var userId = "BE62FBC7-DED9-4420-913A-F47A5324A1C6";
            builder.HttpContext.User = new GenericPrincipal(new GenericIdentity(userId), null);
            builder.HttpContext.Request.Stub(r => r.ServerVariables).Return(serverVariables);
            var controller = new ItinerariesController(new Mock<IActivitiesRepository>().Object, repo.Object, new Mock<IItineraryContainer>().Object);
            builder.InitializeController(controller);

            var view = controller.AddComment(3, "test comment");
            repo.Verify(r => r.AddComment(It.Is<ItineraryComment>(c => c.Body == "test comment")));
            repo.Verify(r => r.AddComment(It.Is<ItineraryComment>(c => c.IpAddress == "127.0.0.1")));
            repo.Verify(r => r.AddComment(It.Is<ItineraryComment>(c => c.ItineraryId == 3)));
            repo.Verify(r => r.AddComment(It.Is<ItineraryComment>(c => c.UserId == new Guid(userId))));
        }

        private static void PrepareControllerMock(Controller controller)
        {
            var controllerContext = new Mock<ControllerContext>();
            var requestContext = new Mock<RequestContext>();
            var routes = new RouteCollection();
            var response = new Mock<System.Web.HttpResponseBase>();
            var request = new Mock<System.Web.HttpRequestBase>();
            var httpContext = new Mock<System.Web.HttpContextBase>();

            controllerContext.CallBase = true;

            requestContext.Setup(r => r.RouteData).Returns(new RouteData());
            requestContext.Setup(r => r.HttpContext).Returns(httpContext.Object);

            httpContext.Setup(h => h.Response).Returns(response.Object);
            httpContext.Setup(h => h.Request).Returns(request.Object);

            controllerContext.Object.RequestContext = requestContext.Object;

            RoutingManager.RegisterRoutes(routes);

            response.Setup(r => r.ApplyAppPathModifier(It.IsAny<string>())).Returns(string.Empty);

            controller.Url = new UrlHelper(requestContext.Object, routes);
            controller.ControllerContext = controllerContext.Object;
        }
    }

    internal class MockActionExport
    {
        public const string TargetSite = "MyTargetSite";
        public static readonly RouteValueDictionary ExportedRouteValueDictionary = new RouteValueDictionary(new { controller = "MyTargetSite" });
        public static readonly ExtensionLink ExportedExtensionLink = new ExtensionLink() { LinkText = "Text" };

        [Export(typeof(RouteValueDictionary))]
        [ExtensionSiteMetadata(TargetExtensionSite = TargetSite)]
        public RouteValueDictionary EmailToolboxExport
        {
            get
            {
                return ExportedRouteValueDictionary;
            }
        }

        [Export(typeof(ExtensionLink))]
        [ExtensionSiteMetadata(TargetExtensionSite = TargetSite)]
        public ExtensionLink EmailLinkExport
        {
            get
            {
                return ExportedExtensionLink;
            }
        }
    }
}
